/* Ghibli API demo with data TRANSFORM
   - Fetch films (no key needed)
   - Transform raw payload into a richer, typed model
   - Render cards + search + sort
*/

const API_URL = "https://ghibliapi.vercel.app/films";

const grid = document.getElementById("grid");
const statusBox = document.getElementById("status");
const searchInput = document.getElementById("searchInput");
const sortSelect = document.getElementById("sortSelect");
const resetBtn = document.getElementById("resetBtn");

let films = [];      // transformed data
let viewFilms = [];  // filtered/sorted view

// ---------- TRANSFORM HELPERS ----------
function toSlug(str = "") {
  return String(str)
    .toLowerCase()
    .trim()
    .replace(/[^a-z0-9\s-]/g, "")
    .replace(/\s+/g, "-")
    .replace(/-+/g, "-");
}

function gradeFromScore(n = 0) {
  if (n >= 90) return "S";
  if (n >= 80) return "A";
  if (n >= 70) return "B";
  if (n >= 60) return "C";
  return "D";
}

function decadeFromYear(yearNum) {
  if (!Number.isFinite(yearNum)) return "Unknown";
  const d = Math.floor(yearNum / 10) * 10;
  return `${d}s`;
}

function tokenize(str = "") {
  return str
    .toLowerCase()
    .split(/[^a-z0-9]+/g)
    .filter(Boolean);
}

/**
 * Transform raw API film into a normalized, enriched object.
 */
function transformFilm(raw) {
  const title = raw.title ?? "Untitled";
  const director = raw.director ?? "Unknown";
  const yearNum = Number(raw.release_date);
  const scoreNum = Number(raw.rt_score);

  return {
    // original fields (kept for compatibility)
    ...raw,

    // normalized fields
    title,
    director,
    release_year: Number.isFinite(yearNum) ? yearNum : null,
    score_numeric: Number.isFinite(scoreNum) ? scoreNum : 0,

    // derived fields
    decade: decadeFromYear(yearNum),
    score_grade: gradeFromScore(scoreNum),
    slug: toSlug(title),

    // improved search field: title + director tokens
    keywords: Array.from(new Set([...tokenize(title), ...tokenize(director)])),
  };
}

/**
 * Transform the whole dataset safely.
 */
function transformFilms(list) {
  if (!Array.isArray(list)) return [];
  return list.map(transformFilm);
}

// ---------- FETCH ----------
async function fetchFilms() {
  setStatus("Loading films…", true);
  try {
    const res = await fetch(API_URL);
    if (!res.ok) throw new Error(`HTTP ${res.status}`);
    const data = await res.json();

    films = transformFilms(data); 
    viewFilms = films.slice();
    render();
    clearStatus();
  } catch (err) {
    console.error(err);
    setStatus("Failed to load films. Check your connection and try again.", false);
  }
}

// ---------- RENDER ----------
function render() {
  grid.innerHTML = "";
  if (viewFilms.length === 0) {
    grid.innerHTML = `<p style="color:#9aa3af">No results.</p>`;
    return;
  }

  const frag = document.createDocumentFragment();
  viewFilms.forEach(film => {
    const card = document.createElement("article");
    card.className = "card";
    card.setAttribute("data-slug", film.slug); // example usage of transformed slug

    const banner = document.createElement("img");
    banner.className = "banner";
    banner.loading = "lazy";
    banner.alt = `${film.title} banner`;
    banner.src = film.movie_banner || film.image || "";
    card.appendChild(banner);

    const content = document.createElement("div");
    content.className = "content";

    const h3 = document.createElement("h3");
    h3.textContent = film.title;
    content.appendChild(h3);

    const meta = document.createElement("div");
    meta.className = "meta";
    meta.innerHTML = `
      <span class="chip">Year: ${film.release_year ?? "—"}</span>
      <span class="chip">Decade: ${film.decade}</span>
      <span class="chip">Director: ${film.director}</span>
      <span class="chip">Score: ${film.score_numeric}</span>
      <span class="chip">Grade: ${film.score_grade}</span>
    `;
    content.appendChild(meta);

    const p = document.createElement("p");
    p.className = "desc";
    p.textContent = film.description || "";
    content.appendChild(p);

    card.appendChild(content);
    frag.appendChild(card);
  });

  grid.appendChild(frag);
}

// ---------- SEARCH / SORT ----------
function applySearch(term) {
  const q = term.trim().toLowerCase();
  if (!q) {
    viewFilms = films.slice();
    return;
  }

  viewFilms = films.filter(f => {
    // searchable across normalized fields + keywords
    return (
      (f.title || "").toLowerCase().includes(q) ||
      (f.director || "").toLowerCase().includes(q) ||
      (f.release_year?.toString() || "").includes(q) ||
      (f.decade || "").toLowerCase().includes(q) ||
      f.keywords.some(k => k.includes(q))
    );
  });
}

function applySort(mode) {
  const byNum = (k, dir = 1) => (a, b) => ((a[k] ?? 0) - (b[k] ?? 0)) * dir;
  const byStr = (k, dir = 1) => (a, b) => (a[k] || "").localeCompare(b[k] || "", undefined, { sensitivity: "base" }) * dir;

  switch (mode) {
    case "title-asc":  viewFilms.sort(byStr("title", +1)); break;
    case "title-desc": viewFilms.sort(byStr("title", -1)); break;
    case "year-asc":   viewFilms.sort(byNum("release_year", +1)); break;
    case "year-desc":  viewFilms.sort(byNum("release_year", -1)); break;
    case "score-asc":  viewFilms.sort(byNum("score_numeric", +1)); break;
    case "score-desc": viewFilms.sort(byNum("score_numeric", -1)); break;
    default: break;
  }
}

// ---------- STATUS ----------
function setStatus(msg, isLoading) {
  statusBox.innerHTML = "";
  statusBox.style.display = "flex";
  if (isLoading) {
    const spin = document.createElement("div");
    spin.className = "spinner";
    spin.setAttribute("aria-hidden", "true");
    statusBox.appendChild(spin);
  }
  const span = document.createElement("span");
  span.textContent = msg;
  statusBox.appendChild(span);
}
function clearStatus(){ statusBox.style.display = "none"; }

// ---------- EVENTS ----------
searchInput.addEventListener("input", () => {
  applySearch(searchInput.value);
  applySort(sortSelect.value);
  render();
});

sortSelect.addEventListener("change", () => {
  applySearch(searchInput.value);
  applySort(sortSelect.value);
  render();
});

resetBtn.addEventListener("click", () => {
  searchInput.value = "";
  sortSelect.value = "title-asc";
  viewFilms = films.slice();
  applySort(sortSelect.value);
  render();
});

// ---------- INIT ----------
fetchFilms();
